#include "global.h"

#include "Port.h"
#include <Preferences.h> //lib pour écrire dans l'eeprom
//#include <arpschuino32_core.h>
#include <arpschuino.h>
#include "custom_port.h"
#include "ArpAccelStepper.h"
#include "arp_server.h"

String autocalibration_status = "autocalibration request";

#ifdef WILULU32
    Port portW('W');
#endif
Port portA('A'); // creation des objets port
#ifdef ARPSCHUINO32
    Port portB('B');
#endif

Preferences port_preferences;

Port::Port(char name) : m_port_name(name)
{

}

///////////////////////////////////////////////////////
////////////////OTHER METHODS/////////////////////////
//////////////////////////////////////////////////////
void Port::find_mode()
{
    port_preferences.begin("port_datas", false);
    uint8_t defaultVal = 0;
    //Serial.println("debug :");
    #ifdef WILULU32
        //Serial.println("WILULU32 defined");
        //Serial.print("m_port_name = "); Serial.println(m_port_name);
        if(m_port_name=='A')
        {
            defaultVal = 255;
            //Serial.println("defaultVal = 255");
        }
    #endif
    m_mode = load("mode", defaultVal);
    //Serial.print("m_mode = "); Serial.println(m_mode);
    //Serial.println("end of debug");
}

void Port::debug()
{
    Serial.print("debug port ");
    Serial.println(m_port_name);
}
///////////////////////////////////////////////////////
////////////////SET METHODS////////////////////////////
//////////////////////////////////////////////////////

void Port::setNbDmxChannels(uint16_t nbChannels) // should auto adjust, except for pixel Led
{
    m_nbDmxChannels = nbChannels;
}
///////////////////////////////////////////////////////
////////////////GET METHODS////////////////////////////
//////////////////////////////////////////////////////

uint16_t Port::getNbDmxChannels() const
{
    return m_nbDmxChannels;
}

uint16_t Port::getMode() const
{
    return m_mode;
}

String Port::writeMode() const
{
    String str;
    switch (getMode())
    {
    case 0:
        str = "PWM";
        break;
    case 1:
        str = "on/off";
        break;
    case 2:
        str = "servo";
        break;
    case 3:
        str = "stepper";
        break;
    case 4:
        str = "CCmotor";
        break;
    case 20:
        str = "inputs";
        break; 
    case 254:
        str = "custom";
        break;               
    default:
        str = "inactive";
    }
    return str;
}
/////////////////////////////////////////////////////////
////////////////RECORD METHODS//////////////////////////
///////////////////////////////////////////////////////
void Port::record(String name, uint8_t val, int num) const
{
    // String str(m_port_name);
    // str += name;
    String str = format_key(name, num);
    const char *key = str.c_str();
    port_preferences.putUChar(key, val);
}

void Port::record(String name, uint16_t val, int num) const
{
    // String str(m_port_name);
    // str += name;
    String str = format_key(name, num);
    const char *key = str.c_str();
    port_preferences.putUShort(key, val);
}

void Port::record(String name, uint32_t val, int num) const
{
    // String str(m_port_name);
    // str += name;
    String str = format_key(name, num);
    const char *key = str.c_str();
    port_preferences.putULong(key, val);
}

void Port::record(String name, bool val, int num) const
{
    // String str(m_port_name);
    // str += name;
    String str = format_key(name, num);
    const char *key = str.c_str();
    port_preferences.putBool(key, val);
}

void Port::record(String name, float val, int num) const
{
    // String str(m_port_name);
    // str += name;
    String str = format_key(name, num);
    const char *key = str.c_str();
    port_preferences.putFloat(key, val);
}
/////////////////////////////////////////////////////////////////////////////
//////////////////////READ METHODS (read preferences)/////////////////////
////////////////////////////////////////////////////////////////////////////

String Port::format_key(String name, int num) const
{
    String Sport(m_port_name);
    String Snum(num);
    if(num==255)
        Snum="";
    return  Sport + Snum + name;
}

uint8_t Port::load(String name, uint8_t val, int num) const//num=255 by default
{
    String str = format_key(name, num);
    const char *key = str.c_str();
    return port_preferences.getUChar(key, val);
}

uint16_t Port::load(String name, uint16_t val, int num) const//num=255 by default
{
    String str = format_key(name, num);
    const char *key = str.c_str();
    return port_preferences.getUShort(key, val);
}

uint32_t Port::load(String name, uint32_t val, int num) const//num=255 by default
{
    String str = format_key(name, num);
    const char *key = str.c_str();
    return port_preferences.getULong(key, val);
}

bool Port::load(String name, bool val, int num) const//num=255 by default
{
    String str = format_key(name, num);
    const char *key = str.c_str();
    return port_preferences.getBool(key, val);
}

float Port::load(String name, float val, int num) const//num=255 by default
{
    String str = format_key(name, num);
    const char *key = str.c_str();
    return port_preferences.getFloat(key, val);
}

///////////////////////////////////////////////////////
//////////////////SETUP METHODS////////////////////////
///////////////////////////////////////////////////////
void Port::init()
{
    //Serial.println("init !");
    #ifdef WILULU32
        m_nb_Pins = (m_port_name == 'W') ? NB_WILULU_OUT : 8;
    #endif
    #ifdef ARPSCHUINO32
        m_nb_Pins = 8;
    #endif    
    m_startPin = (m_port_name == 'A') ? 0 : 8;////////////////
    Serial.print("Port ");
    Serial.print(m_port_name);
    Serial.print(": ");Serial.print(writeMode());
    
    //
    
    Serial.println(" ");
    switch (m_mode)
    {
        case 0:
        { // pwm
            PWM_setup();
            break;
        }
        case 1:
        {
            on_off_setup();
            break;
        }
        case 2:
        {
            servo_setup();
            break;
        }
        case 3:
        {
            stepper_setup();
            break;
        }
        case 4:
        {
            CCmotor_setup();
            break;
        }
        case 20:
        {
            inputs_setup();
            break;
        } 
        case 254:
        {
            custom_setup();
            break;
        }
        default:
        {
            //Serial.println(": inactive...");
        } 
        
    }
    Serial.println();            //Serial.println();  
}

void Port::PWM_setup()
{
    int32_t freq = 40000000;
    //uint8_t defaultVal = 14;
    m_pwmRes = load("PWM_resolution", DFLT_PWM_resolution);////////////////////////////////////////////////////////
    //Serial.print("la resolutions est :");Serial.println(m_pwmRes);
    for (uint8_t i = 1; i < m_pwmRes; i++)
    {
        freq = freq / 2;
    }
    //uint8_t startPin = (m_port_name == 'A') ? 0 : 8;
    m_dmxRes = load("dmxRes", DFLT_dmxRes);
    if (m_dmxRes)
    {
        m_nbDmxChannels = m_nb_Pins*2;
    }
    else
    {
        m_nbDmxChannels = m_nb_Pins;
    }
    m_pwmCurve = load("pwmCurve", DFLT_pwmCurve);
    for (uint8_t i = m_startPin; i < m_startPin + m_nb_Pins; i++)
    {
        // pinMode(Arp[i], OUTPUT);
        ledcSetup(i, freq, m_pwmRes); // configure LED PWM functionalitites
        ledcAttachPin(Arp[i], i);     // attach the channel to the GPIO to be controlled
    }
    //Serial.print(": PWM ");
    Serial.print(m_pwmRes);
    Serial.print(" bits; freq ");
    Serial.print(freq);
    Serial.print(" hz; ");
    Serial.print((m_pwmCurve == true) ? "straight" : "LED");////////////////////
    Serial.print(" curve; ");
    if (m_dmxRes)
        Serial.print(", 16 bits DMX input");
    Serial.println();
}

void Port::on_off_setup()
{
    m_inverted = load("inverted", DFLT_inverted);
    m_trigger = load("trigger", DFLT_trigger);
    m_nbDmxChannels = m_nb_Pins;
    for (int i = m_startPin; i < m_startPin + m_nb_Pins; i++)
    {
        pinMode(Arp[i], OUTPUT);
        if (m_inverted)
            digitalWrite(Arp[i], HIGH);
    }
    Serial.print(" trigger at ");
    Serial.print(m_trigger);
    Serial.print(" (");
    Serial.print((m_trigger * 100) / 255);
    Serial.print("%)");
    if (m_inverted)
        Serial.print(", inverted");
    Serial.println();
}

void Port::servo_setup()
{
    //const uint8_t defaultAngle = 160;
    m_nbDmxChannels = 0;
    for (int i = 0; i < 8; i++)
    {
        m_servo_active[i] = load("servoActive", DFLT_Active, i);
        if(m_servo_active[i])
            m_nbDmxChannels ++;
    }
    m_servo = new Servo[m_nbDmxChannels];//create the needed number of servos
    uint8_t servo_num=0;
    for (int i = 0; i < 8; i++)
    {
        //m_servo_active[i] = load("servoActive", true, i);
        Serial.print("servo ");
        Serial.print(i);        
        if (m_servo_active[i])
        {
            m_servoAngle[i] = load("angle", DFLT_angle,i);
            m_servoInverted[i] = load("inverted", DFLT_inverted,i);
            m_servoDetach[i] = load("detach", DFLT_detach,i);
            if (m_servoDetach[i] == false)
            {
                uint8_t output = i + m_startPin;
                m_servo[servo_num].attach(Arp[output], output); //(pin,channel)
            }
            Serial.print(": ");
            Serial.print(m_servoAngle[i]);
            Serial.print("°");
            if (m_servoInverted[i])
                Serial.print(", inverted");
            if (m_servoDetach[i])
                Serial.print(", mode detach");
            Serial.println();
            servo_num++;
        }
        else
        {
            Serial.println(" inactive; ");
        } 
    }
    //Serial.print("m_nbDmxChannels servo : ");Serial.println(m_nbDmxChannels);///////////debug/////////////////
}

void Port::stepper_setup()
{
    uint8_t nb_stp;
    /////////////////////////////////////////////////////
    m_stp_driver_type = load("stpDrivMode", DFLT_stpDrivMode);//ArpAccelStepper::DRIVER;
    ///////////////////////////////////////////////////////
    switch (m_stp_driver_type) {
        case ArpAccelStepper::DRIVER:
            //m_stp_driver_type = ArpAccelStepper::DRIVER;    // DRIVER
            nb_stp=4;
            Serial.println(" stp/dir driver ");
            break;
        case ArpAccelStepper::FULL4WIRE:
            //m_stp_driver_type = ArpAccelStepper::FULL4WIRE;    // FULL4WIRE
            nb_stp=2;
            Serial.println(" unipolar full step ");
            break;
        case ArpAccelStepper::HALF4WIRE:
            //m_stp_driver_type = ArpAccelStepper::HALF4WIRE;   
            nb_stp=2;
            Serial.println(" unipolar half step ");
            break;
        default:
            Serial.println("invalid driver type!");
            //m_stp_driver_type = ArpAccelStepper::DRIVER;    // DRIVER
            nb_stp=4;           
    } 

    m_nbDmxChannels = 0;

    for (int i = 0; i < nb_stp; i++)
    {
        m_stp_active[i] = load("stpActive", true,i);
        if(m_stp_active[i])
            m_nbDmxChannels = m_nbDmxChannels+2;
    }    

    m_stepper = new ArpAccelStepper[nb_stp];
    bool homing[nb_stp]={false};

    for (int i = 0; i < nb_stp; i++)
    {
        Serial.print("stepper ");
        Serial.print(i);
        if (m_stp_active[i])
        {
            //m_stepper[i].setMinPulseWidth(50);
            uint16_t stepPerRevolution = load("stpPerRevol", DFLT_SPR,i);
            m_stepper[i].set_interface(m_stp_driver_type);//ArpAccelStepper::DRIVER
            if(nb_stp==4)
            {
                m_stepper[i].attach(Arp[i * 2 + 1 + m_startPin], Arp[i * 2 + m_startPin], stepPerRevolution);
            }    
            else if(nb_stp==2)
            {
                m_stepper[i].attach(Arp[i*4 +m_startPin], Arp[i*4 + 2+m_startPin], Arp[i*4 + 1+m_startPin], Arp[i*4 + 3+m_startPin], stepPerRevolution);
            }
            m_stepper[i].setMinPulseWidth(20);/////////////////////////m_stepper[i].setMinPulseWidth(20);

            Serial.print(" steps per revolution: ");
            Serial.print(stepPerRevolution);
            Serial.print("; ");

            uint16_t rpmMin = load("_rpm_min", DFLT_Min,i);
            Serial.print(" rpmmin: ");
            Serial.print(rpmMin);
            Serial.print("; ");

            uint16_t rpmMax = load("_rpm_max", DFLT_Max,i);
            Serial.print(" max: ");
            Serial.print(rpmMax);
            Serial.print("; ");

            float revol = load("_revol", DFLT_Revol,i);
            Serial.print(" revolutions: ");
            Serial.print(revol);
            Serial.print("; ");

            m_endPosition[i] = stepPerRevolution * revol;
            // Serial.print(" m_endPosition: ");//////////////////////////////
            // Serial.print(m_endPosition[i]);/////////////////////////////////////////
            // Serial.print("; ");//////////////////////////////////////////            

            float acceleration = load("_accel", DFLT_Accel,i);
            Serial.print(" acceleration: ");
            Serial.print(acceleration);
            Serial.print("; ");            

            m_action[i] = load("stpAction", DFLT_Action,i);//variable action dans port.h
            Serial.print(" action: ");
            uint16_t dmxRes = _8bits;

            if (m_action[i] == 0)
            {
                Serial.print("perform");
                Serial.print("; ");
                Serial.print(" DMX resolution: ");

                m_stp_dmxRes[i] = load("stp_dmxRes", DFLT_dmxRes,i);
                if (m_stp_dmxRes[i] == true)
                {
                    dmxRes = _16bits;
                    Serial.print("16bits");
                    m_nbDmxChannels++;
                }
                else
                {
                    Serial.print("8bits");
                }
                Serial.print("; ");
            }
            else
            {
                Serial.print("continuous; ");
            }
            m_stepper[i].init(rpmMin, rpmMax, revol, dmxRes,m_action[i],acceleration);

            bool inverted = load("stpInverted", DFLT_inverted,i);
            if (inverted)
            {
                m_stepper[i].invert_rotation(true);
                Serial.print(" inverted; ");
            }
            m_stpZeroSW[i] = load("stpZeroSW", DFLT_SWITCH,i);

            uint8_t homSpeed;

            if(m_stpZeroSW[i])
            {
                Serial.print(", Zero switch");
                m_stpzero_input[i] = load("stpZero_i", DFLT_input,i);
                if(m_stpzero_input[i]<255)
                {
                    if (m_stpzero_input[i]<16)
                        pinMode(Arp[m_stpzero_input[i]],INPUT_PULLUP);
                    else if(m_stpzero_input[i]<20)
                        pinMode(Arp[m_stpzero_input[i]],INPUT);
                    Serial.print(": Arp");
                    Serial.print(m_stpzero_input[i]);
                    Serial.print("; ");
                }
                else Serial.print(": no pin assigned"); 

                homSpeed = rpmMax/load("stpHomSpeed", DFLT_stpHomSpeed,i);

                homing[i] = load("stpHomingSW", DFLT_stpHomingSW,i);
                if(homing[i])
                {
                    Serial.print(" homing at startup; ");
                    m_stepper[i].setMaxSpeed(m_stepper[i].RPMtoSPS (homSpeed));//speed for homing
                }
            }
            m_stpEndSW[i] = load("stpEndSW", DFLT_SWITCH,i);
           
            if(m_stpEndSW[i])
            {
                Serial.print(", End switch");
                m_stpend_input[i] = load("stpEnd_i", DFLT_input,i);
                if(m_stpend_input[i]<255)
                {
                    if (m_stpzero_input[i]<16)
                        pinMode(Arp[m_stpend_input[i]],INPUT_PULLUP);
                    else if(m_stpzero_input[i]<20)
                        pinMode(Arp[m_stpend_input[i]],INPUT);
                    //attachInterrupt(m_stpzero_input[i], fonction_ISR, FALLING);///////////////////////////////////////

                    Serial.print(": Arp");
                    Serial.print(m_stpend_input[i]);
                }
                else Serial.print(": no pin assigned"); 
            }                   
            Serial.println();
            ///////////////// autocalibration /////////////////////////////
            //String num(i);////////////////////////////////////////////////////////////////////////////////////////////////////////////
            //String str = num + "autocalib";//stepperNum + "autocalibration"//////////////////////////////////////////////////
            //const char *word = str.c_str();//////////////////////////////////////////////////////////////////////////
            //bool default_val=false;
            bool do_autocal = load("autocalib", DFLT_autocalib,i);
            //Serial.println(do_autocal);
         
            if(do_autocal)
            {
                Serial.print("autocalibrating ");
                record("autocalib", false,i);

                m_stepper[i].setMaxSpeed(m_stepper[i].RPMtoSPS (homSpeed));//speed for homing
                m_stepper[i].move(0x6FFFFFFF);//go to the end
                autocalibration_status = "go to the end switch";
                while(digitalRead(Arp[m_stpend_input[i]]))//end switch not pressed
                {   
                    // Print dots while waiting for homing to finish
                    static uint16_t dot=0;
                    if(dot==0)  
                        Serial.print('.');
                    dot++;
                    ////////////////////////////////////////////////
                    m_stepper[i].run(); 
                }
                Serial.print("stepper");Serial.print(i);Serial.print(" at end !");
                m_stepper[i].setCurrentPosition(0);
                m_stepper[i].move(-0x6FFFFFFF); //go back
                autocalibration_status = "back to the zero switch";
                while(digitalRead(Arp[m_stpzero_input[i]]))//zero switch not pressed
                {   
                    // Print dots while waiting for homing to finish
                    static uint16_t dot=0;
                    if(dot==0)  
                        Serial.print('.');
                    dot++;
                    ////////////////////////////////////////////////
                    m_stepper[i].run(); 
                }
                Serial.print("stepper");Serial.print(i);Serial.println(" at zero !");
                m_endPosition[i] = abs(m_stepper[i].currentPosition());
                m_stepper[i].setCurrentPosition(0);
                autocalibration_status = "autocalibration complete";
                Serial.print("autocalibrating complete, ");
                revol = ((float)m_endPosition[i]/stepPerRevolution);

                record("_revol", revol,i);
                Serial.print("new revolutions value: ");
                Serial.println(revol);

                m_stepper[i].init(rpmMin, rpmMax, revol, dmxRes,m_action[i],acceleration);
            }

        }
        else
        {
            Serial.println(" inactive; ");
        }        
    } 
    /////////////////////////////////////// homing ////////////////////////
    if(homing[0] || homing[1] || homing[2] || homing[3])
    {
        Serial.print("homing port ");Serial.print(m_port_name);
        bool atHome[4]={false};      
        while(not atHome[0] || not atHome[1] || not atHome[2] || not atHome[3])
        {   
            static uint16_t dot=0;
            if(dot==10000)
            { 
                Serial.print('.');
                dot=0;
            }     
            dot++;

            for(int i=0;i<4;i++)
            {
                if(homing[i] && not atHome[i])
                {
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                    int32_t destination=-0x6FFFFFFF;
                    //bool inverted = load("stpInverted", DFLT_inverted,i);                    
                    if(m_stepper[i].inverted())
                        destination=destination * -1;  //this is not a pointer ! destination multiplicated by -1
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////

                    m_stepper[i].move(destination); 
                    if(m_stpzero_input[i]!=255 && digitalRead(Arp[m_stpzero_input[i]]))//switch not pressed
                    {
                        m_stepper[i].run();
                    }    
                    else //switch pressed
                    {
                        Serial.print("stepper");Serial.print(i);Serial.print(" at home ! ");
                        m_stepper[i].setCurrentPosition(0);
                        atHome[i]=true;
                    }                    
                    
                }
                else atHome[i]=true;
            }
        }
    }
    Serial.println();
}

void Port::CCmotor_setup()
{
    //uint8_t default_CC_driver_type = 0;
    //uint8_t default_CC_threshold = 1;
    //uint8_t default_input =255;
    //uint16_t default_CC_freq = 25000;//m_CC_freq[4]
    //m_nbDmxChannels = m_nb_Pins;

    for (int i = 0; i < 4; i++)
    {
        m_CC_active[i] = load("CCActive", DFLT_Active,i);
        Serial.print("DC motor ");
        Serial.print(i);
        if (m_CC_active[i])
        {
            m_nbDmxChannels = m_nbDmxChannels+2;//////////////////////////////////////////////////////
            m_CC_driver_type[i] = load("CCdvrType", DFLT_CC_driver_type,i);

            Serial.print((m_CC_driver_type[i] == 0) ? " H-bridge classic" : " speed/dir");            

            m_cc_threshold[i] = load("thresold", DFLT_CC_threshold,i);
            Serial.print(", thresold: ");
            Serial.print(m_cc_threshold[i]);

            uint8_t percent = m_cc_threshold[i] * 100 / 255;
            Serial.print(" (");
            Serial.print(percent);
            Serial.print("%)");

            //default_CC_freq = 25000;//m_CC_freq[4]
            m_CC_freq[i] = load("CC_freq", DFLT_CC_freq,i);
            Serial.print(", PWM frequency: ");
            Serial.print(m_CC_freq[i]);

            m_CC_inverted[i] = load("CCInverted", DFLT_inverted,i);
            Serial.print(m_CC_inverted[i]? ", inverted":"");

            m_CCZeroSW[i] = load("CCZeroSW", DFLT_SWITCH,i);
            if(m_CCZeroSW[i])
            {
                Serial.print(", Zero switch");
                m_CCzero_input[i] = load("CCZero_i", DFLT_input,i);
                if(m_CCzero_input[i]<255)
                {
                    pinMode(Arp[m_CCzero_input[i]],INPUT_PULLUP);
                    Serial.print(": Arp");
                    Serial.print(m_CCzero_input[i]);
                }
                else Serial.print(": no pin assigned");           
            }
            m_CCEndSW[i] = load("CCEndSW", DFLT_SWITCH,i);
            if(m_CCEndSW[i])
            {
                Serial.print(", End switch");
                m_CCend_input[i] = load("CCEnd_i", DFLT_input,i);
                if(m_CCend_input[i]<255)
                {
                    pinMode(Arp[m_CCend_input[i]],INPUT_PULLUP);
                    Serial.print(": Arp");
                    Serial.print(m_CCend_input[i]);
                }
                else Serial.print(": no pin assigned"); 
            }
            uint8_t chan = 2*i + m_startPin;
            for (int j=0; j <2; j++)
            {
                //Serial.println(chan+j);
                ledcSetup(chan+j, m_CC_freq[i], 8);   // channel, freq, Resolution
                ledcAttachPin(Arp[chan+j], chan+j); // attach the channel to the GPIO to be controlled 
            }               
        }
        else
        {
            Serial.print(" inactive");
        }
        Serial.println(); 
    }
    //Serial.print("m_nbDmxChannels CC : ");Serial.println(m_nbDmxChannels);///////////debug/////////////////


    // for (int i = m_startPin; i < m_startPin + 8; i++)
    // {
    //     ledcSetup(i, 25000, 8);   // channel, freq, Resolution,25000Hz should be unaudible
    //     ledcAttachPin(Arp[i], i); // attach the channel to the GPIO to be controlled
    // }
}

void Port::inputs_setup()
{
    m_nbDmxChannels = 0;
    for (int i = m_startPin; i < m_startPin + 8; i++)
    {
        pinMode(Arp[i], INPUT_PULLUP);
    }
    Serial.println();
}

void Port::custom_setup()
{
    #ifdef WILULU32
        if (m_port_name == 'W')
        {
            setupW();
        }
    #endif    
    if (m_port_name == 'A')
    {
        setupA();
    }
    #ifdef ARPSCHUINO32
        else if (m_port_name == 'B')
        {
            setupB();
        }
    #endif
}

///////////////////////////////////////////////////////
//////////////////ACTION METHODS////////////////////////
///////////////////////////////////////////////////////
void Port::action(const std::vector<uint8_t> &DMXslice)
{
    //uint8_t startPin = (m_port_name == 'A') ? 0 : 8;
                    //Serial.print("ch1 ");Serial.print(DMXslice[0]);Serial.print(" ch2");Serial.println(DMXslice[1]);
    //Serial.println(DMXslice[0]);
    switch (m_mode)
    {
        case 0:
        {
            PWM_action(DMXslice);
            break;
        }
        case 1:
        {
            on_off_action(DMXslice);
            break;
        }
        case 2:
        {
            servo_action(DMXslice);
            break;
        }
        case 3:
        {
            stepper_action(DMXslice);
            break;
        }
        case 4:
        {
            CCmotor_action(DMXslice);
            break;
        }
        case 254:
        {
            custom_action(DMXslice);
        }
    }
}

void Port::PWM_action(const std::vector<uint8_t> &DMXslice)
{
    if (m_dmxRes)
    { // if 16 bits DMX
        if (m_pwmCurve == 0)
        { // if led curve
            for (int i = 0; i < m_nb_Pins; ++i)
            {
                m_target_level[i] = DMXslice[i * 2] << 8 | DMXslice[i * 2 + 1];             // level= DMXslice[i*2] << 8 | DMXslice[i*2+1];
                uint32_t level32 = ((m_target_level[i] + 1) * (m_target_level[i] + 1)) - 1; // uint32_t level32 =((level+1)*(level+1))-1;
                m_target_level[i] = level32 >> (32 - m_pwmRes);                             // level = level32 >>(32-m_pwmRes);
            }
        }
        else
        { // if straight curve
            for (int i = 0; i < m_nb_Pins; ++i)
            {
                m_target_level[i] = DMXslice[i * 2] << 8 | DMXslice[i * 2 + 1]; // level= DMXslice[i*2] << 8 | DMXslice[i*2+1];
            }
        }
    }
    else
    { // if 8 bits DMX
        if (m_pwmCurve == 0)
        { // if led curve
            for (int i = 0; i < m_nb_Pins; ++i)
            {
                m_target_level[i] = (((DMXslice[i] + 1) * (DMXslice[i] + 1) - 1) >> (16 - m_pwmRes)); // level = (((DMXslice[i]+1)*(DMXslice[i]+1)-1)>>(16-m_pwmRes));
            }
        }
        else
        { // if straight curve
            for (int i = 0; i < m_nb_Pins; ++i)
            {
                m_target_level[i] = DMXslice[i] << (m_pwmRes - 8); // level= DMXslice[i]<<(m_pwmRes-8);
            }
        }
    }
}

void Port::on_off_action(const std::vector<uint8_t> &DMXslice) const
{
    for (int i = 0; i < m_nbDmxChannels; ++i)
    {
        if (DMXslice[i] > m_trigger - 1)
        {
            digitalWrite(Arp[m_startPin + i], 1 - m_inverted);
        }
        else
        {
            digitalWrite(Arp[m_startPin + i], m_inverted);
        }
    }
    //Serial.print("on off");
}

void Port::servo_action(const std::vector<uint8_t> &DMXslice) 
{
    uint8_t this_servo_addr = 0;
    static uint32_t previousMillis[8]={0};
    //static uint8_t oldVal[8]={0};
    for (int i = 0; i < 8; ++i)
    {
        if (m_servo_active[i])
        {
            uint8_t min = 0, max = m_servoAngle[i];
            if (m_servoInverted[i])
            {
                min = m_servoAngle[i], max = 0;
            }
            if (m_servoDetach[i])
            {
                if (DMXslice[this_servo_addr] != m_oldVal[i])
                {
                    uint8_t output = i + m_startPin;
                    m_servo[this_servo_addr].attach(Arp[output], output); //(pin,channel)
                    m_servo[this_servo_addr].write(map(DMXslice[this_servo_addr], 0, 255, min, max));
                    previousMillis[i] = millis();
                }
                else if (m_servo[i].attached())
                {
                    if((millis() - previousMillis[i] > 1200))
                    {
                        m_servo[this_servo_addr].detach();
                    }
                }   
                m_oldVal[i] = DMXslice[this_servo_addr];
            }
            else
            {
                m_servo[this_servo_addr].write(map(DMXslice[this_servo_addr], 0, 255, min, max));

            }
            this_servo_addr++;
        }
    }
}

void Port::stepper_action(const std::vector<uint8_t> &DMXslice) const
{
    //Serial.println(m_stp_driver_type);////////////////////////////////////////////////////////
    uint8_t this_stepper_addr = 0;
    //uint8_t stepper_num = 0;
    for (int i = 0; i < 4; ++i)
    {
        if (m_stp_active[i])
        { 
            //Serial.print(m_port_name); Serial.println(stepper_num);           
            if (m_stp_dmxRes[i]) // 16 Bits
            {
                uint16_t position = ((DMXslice[this_stepper_addr] << 8) | DMXslice[this_stepper_addr + 1]);
                m_stepper[i].refresh(position, DMXslice[this_stepper_addr + 2],m_action[i]);
                this_stepper_addr = this_stepper_addr + 3;
            }
            else // 8 Bits
            {
                m_stepper[i].refresh(DMXslice[this_stepper_addr], DMXslice[this_stepper_addr + 1],m_action[i]);
                this_stepper_addr = this_stepper_addr + 2;
            }
        }
    }
}

void Port::CCmotor_action(const std::vector<uint8_t> &DMXslice) const
{
    uint8_t this_CC_addr = 0;
    for (int i = 0; i < 4; i++)
    {
        if (m_CC_active[i])
        {
            uint8_t speed = DMXslice[this_CC_addr + 1];

            if (speed > 0)
            {
                speed = map(speed, 1, 255, m_cc_threshold[i], 255); // map(value, fromLow, fromHigh, toLow, toHigh)
                //to be optimised
                bool direction = (DMXslice[this_CC_addr] > 127) ? 0 : 1;
                direction = direction - m_CC_inverted[i];

                if (direction) // (dmx level<128) en avant !
                {
                    if(m_CCEndSW[i])
                    {
                        if(!digitalRead(Arp[m_CCend_input[i]]))//end switch pressed
                            speed=0;
                    }
                    if(m_CC_driver_type[i]==0)//H-bridge classic
                    {
                        ledcWrite(m_startPin + i * 2, speed);
                        ledcWrite(m_startPin + i * 2 + 1, 0);                                
                    }                            

                    else if(m_CC_driver_type[i]==1)//speed/dir driver
                    {
                        ledcWrite(m_startPin + i * 2, 0);//DIR
                        ledcWrite(m_startPin + i * 2 + 1, speed);
                    }                            
                }
                else //(dmx level>127) arrière
                {
                    if(m_CCZeroSW[i])
                    {
                        if(!digitalRead(Arp[m_CCzero_input[i]]))//zero switch pressed
                            speed=0;
                    } 
                          
                    if(m_CC_driver_type[i]==0)
                    {                                   
                        ledcWrite(m_startPin + i * 2, 0);
                        ledcWrite(m_startPin + i * 2 + 1, speed);
                    }
                    else if(m_CC_driver_type[i]==1)
                    {
                        ledcWrite(m_startPin + i * 2, 255);
                        ledcWrite(m_startPin + i * 2 + 1, speed);
                    }                            
                }
            }
            else //if (m_CC_driver_type[i]==0)// Stop (freinage)
            {
                ledcWrite(m_startPin + i * 2, 0);
                ledcWrite(m_startPin + i * 2 + 1, 0);
            }
            this_CC_addr = this_CC_addr + 2;
        }
    }
}

void Port::custom_action(const std::vector<uint8_t> &DMXslice) const
{
    #ifdef WILULU32
        if (m_port_name == 'W')
        {
            action_W(DMXslice);
        }
    #endif
    if (m_port_name == 'A')
    {
        action_A(DMXslice);
    }
    #ifdef ARPSCHUINO32
        else if (m_port_name == 'B')
        {
            action_B(DMXslice);
        }
    #endif
}

void Port::runAtEachLoop()
{
    switch (m_mode)
    {
        case 0:
        { // PWM
            for (uint8_t i = 0; i < m_nb_Pins; i++)
            {
                int32_t level;
                if (m_actual_level[i] < m_target_level[i])
                {
                    level = m_actual_level[i] + 1 + (m_actual_level[i] / 256); // m_actual_level[i]++;
                    if (level > m_target_level[i])
                        m_actual_level[i] = m_target_level[i];
                    else
                        m_actual_level[i] = level;
                    ledcWrite(m_startPin + i, m_actual_level[i]);
                }
                else if (m_actual_level[i] > m_target_level[i])
                {
                    level = m_actual_level[i] - 1 - (m_actual_level[i] / 256); // m_actual_level[i]--;
                    if (level < m_target_level[i])
                        m_actual_level[i] = m_target_level[i];
                    else
                        m_actual_level[i] = level;
                    ledcWrite(m_startPin + i, m_actual_level[i]);
                }
            }
            break;
        }
        case 3:
        { // stepper
            for (uint8_t i = 0; i < 4; i++)
            {
                if (m_stp_active[i])
                {
                        //Serial.println(m_stp_driver_type);////////////////////////////////////////////////////////

                    if(m_stpZeroSW[i] && m_stepper[i].distanceToGo()<0)
                    {
                        if(digitalRead(Arp[m_stpzero_input[i]]))
                        {
                            m_stepper[i].run();
                            //Serial.println("run");
                        }    
                        else m_stepper[i].setCurrentPosition(0);
                    }     
                    else if(m_stpEndSW[i] && m_stepper[i].distanceToGo()>0)
                    {
                        if(digitalRead(Arp[m_stpend_input[i]]))
                        {
                            m_stepper[i].run();
                            //Serial.println("run");
                        }    
                        else 
                        {
                            m_stepper[i].setCurrentPosition(m_endPosition[i]);
                            // Serial.print("Current Position: ");//////////////////////////////
                            // Serial.print(m_stepper[i].currentPosition());////////////
                            // //////////////////////////////
                            // Serial.print(m_endPosition[i]);/////////////////////////////////////////
                            // Serial.println("; ");////////////////////////////////////////// 
                        }    
                    }
                    else 
                    {
                        m_stepper[i].run();
                        //Serial.print(i);Serial.println(" run");
                        //Serial.println(m_stepper[i].currentPosition());////////////
                    }                    
                }
            }
            break;
        }
        case 4:
        { // CC motor
            for (uint8_t i = 0; i < 4; i++)
            {
                if (m_CC_active[i])
                {
                    switch (m_CC_driver_type[i])
                    {
                        case 0://0=Hbridge classic
                        {
                            if(m_CCZeroSW[i])// && ledcRead(m_startPin+i))//ledcRead(uint8_t chan)
                            {
                                if(!digitalRead(Arp[m_CCzero_input[i]]))    //zero switch pressed
                                {
                                    ledcWrite(m_startPin + i * 2 + 1, 0);   //stop motor if backward
                                } 
                            }       
                            if(m_CCEndSW[i])// && !ledcRead(m_startPin+i))
                            {
                                if(!digitalRead(Arp[m_CCend_input[i]]))     //end switch pressed
                                {
                                    ledcWrite(m_startPin + i * 2, 0);       //stop motor if forward
                                }   
                            }
                            break;
                        }
                        case 1:
                        {
                            if(m_CCZeroSW[i] && ledcRead(m_startPin+i)>0)//zero switch enable & backward
                            {
                                if(!digitalRead(Arp[m_CCzero_input[i]]))    //zero switch pressed
                                {
                                    ledcWrite(m_startPin + i * 2 + 1, 0);   //stop motor 
                                } 
                            }       
                            if(m_CCEndSW[i] && ledcRead(m_startPin+i)==0)//end switch enable & forward  //direction()
                            {
                                if(!digitalRead(Arp[m_CCend_input[i]]))     //end switch pressed
                                { 
                                    //Serial.print(".");
                                    ledcWrite(m_startPin + i * 2 + 1, 0);       //stop motor
                                }   
                            }  
                            break;
                        }
                    } 
                }                 
            }
            break;
        }    
        case 254:
        {
            #ifdef WILULU32
                if (m_port_name == 'W')
                {
                    at_each_loop_W();
                }
            #endif        
            if (m_port_name == 'A')
            {
                at_each_loop_A();
            }
            #ifdef ARPSCHUINO32
                else if (m_port_name == 'B')
                {
                    at_each_loop_B();
                }
            #endif
            break;
        }
        default : 
        {
            //do nothing...
            //Serial.println("dflt");//debug
        }
    }
}


void Port::stpHome(uint8_t stpNum, uint8_t speed)
{

}

